
/**
 * Plugin.h - Grim Dawn Hook Plugin SDK (c) EliteMMO Network, 2015
 *
 * GDHook is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GDHook is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GDHook.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __GDHOOK_SDK_PLUGIN_H_INCLUDED__
#define __GDHOOK_SDK_PLUGIN_H_INCLUDED__

#if defined (_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// GDHOOK_INTERFACE_VERSION
//
// Defines the current interface version of this SDK. 
// DO NOT EDIT THIS!
//
// Prevents outdated plugins from being loaded on newer versions of
// the GDHOOK project that may have changed the plugin interface.
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#define GDHOOK_INTERFACE_VERSION 1.01

/**
 * @brief Define DirectInput Version
 */
#ifndef DIRECTINPUT_VERSION
#define DIRECTINPUT_VERSION 0x0800
#endif

#include <Windows.h>
#include <functional>
#include <string>

/**
 * @brief Direct3D Includes
 */
#include "d3d9\d3d9.h"
#include "d3d9\d3dx9.h"
#include "d3d9\dinput.h"

/**
 * @brief SDK Includes
 */
#include "newIDirectInputDevice8A.h"
#include "EDK\ECommandParse.h"
#include "EDK\EData.h"
#include "EDK\EException.h"
#include "EDK\EMemory.h"
#include "EDK\EThreading.h"

/**
 * @brief Button click callback function typedef.
 */
typedef std::function<void(int, void*, float, float)> MOUSEEVENT;

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Grim Dawn Lua Hook Definitions
//
// Useful definitions used for Grim Dawn Lua Hook.
//
///////////////////////////////////////////////////////////////////////////////////////////////////

#define HOOK_DIKEYDOWN  0x100 // (wParam = DIKEY Code, lParam = repeat count)
#define HOOK_DIKEYUP    0x101 // (wParam = DIKEY Code, lParam = null)

#define C_REMOVE        "\\cr"                      // Removes the current custom color and resets to the objects default.
#define C_WHITE         "\\cs(255,255,255,255)"     // White
#define C_BLACK         "\\cs(255,0,0,0)"           // Black
#define C_GREY          "\\cs(255,128,128,128)"     // Grey
#define C_RED           "\\cs(255,255,0,0)"         // Red
#define C_DARKRED       "\\cs(255,128,0,0)"         // Dark Red
#define C_PINK          "\\cs(255,255,128,255)"     // Pink
#define C_MAGENTA       "\\cs(255,255,0,128)"       // Magenta
#define C_ORANGE        "\\cs(255,255,128,0)"       // Orange
#define C_YELLOW        "\\cs(255,255,255,0)"       // Yellow
#define C_GREEN         "\\cs(255,0,255,0)"         // Green
#define C_DARKGREEN     "\\cs(255,0,128,0)"         // Dark Green
#define C_LIME          "\\cs(255,128,255,0)"       // Lime Green
#define C_BLUE          "\\cs(255,0,0,255)"         // Blue
#define C_DARKBLUE      "\\cs(255,0,0,128)"         // Dark Blue
#define C_CYAN          "\\cs(255,0,255,255)"       // Cyan

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Grim Dawn Lua Hook Enumerations
//
// Useful enumerations used for Grim Dawn Lua Hook.
//
///////////////////////////////////////////////////////////////////////////////////////////////////

namespace GDHook
{
    namespace Enums
    {
        /**
         * @brief Frame Anchor Points
         */
        enum FrameAnchor : unsigned int
        {
            TopLeft = 0,
            TopRight = 1,
            BottomLeft = 2,
            BottomRight = 3,
            Right = 1,
            Bottom = 2,
        };

        /**
         * @brief Mouse Input Type
         */
        enum MouseInput : unsigned int
        {
            // Click Events
            LeftClick = 0,
            RightClick = 1,
            MiddleClick = 2,
            X1Click = 3,
            X2Click = 4,

            // Scroll Events
            MouseWheelUp = 5,
            MouseWheelDown = 6,

            // Mouse Move Events
            MouseMove = 7,
        };
    }; // namespace Enums
}; // namespace GDHook

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// IConfigurationManager
//
// Configuration interface object to expose helpful functions related to configuration files.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IConfigurationManager
{
    virtual bool LoadConfiguration(const char* className) = 0;
    virtual bool LoadConfigurationEx(const char* className, const char* file) = 0;
    virtual void RemoveConfiguration(const char* className) = 0;
    virtual bool SaveConfiguration(const char* className) = 0;

    virtual const char* GetConfigString(const char* className, const char* name, const char* defaultValue) = 0;
    virtual bool GetConfigBool(const char* className, const char* name, bool defaultValue) = 0;
    virtual int GetConfigInt(const char* className, const char* name, int defaultValue) = 0;
    virtual unsigned int GetConfigUInt(const char* className, const char* name, unsigned int defaultValue) = 0;
    virtual float GetConfigFloat(const char* className, const char* name, float defaultValue) = 0;
    virtual double GetConfigDouble(const char* className, const char* name, double defaultValue) = 0;

    virtual void SetConfigValue(const char* className, const char* name, const char* value) = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Keyboard Callback Typedefs
//
///////////////////////////////////////////////////////////////////////////////////////////////////
typedef HRESULT(__stdcall *GetDataCallback)(DWORD, LPDIDEVICEOBJECTDATA, DWORD, LPDWORD, DWORD);
typedef HRESULT(__stdcall *GetStateCallback)(DWORD, LPVOID);
typedef BOOL(__stdcall *KeyboardProcCallback)(WPARAM, LPARAM);

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// IKeyboard
//
// Keyboard interface object to expose helpful functions related to the keyboard.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IKeyboard : newIDirectInputDevice8A
{
    // The following are exposed from newIDirectInputDevice8A:
    //      HWND GetParentWindow(void);
    //      bool GetBlocked(void);
    //      void SetBlocked(bool bBlocked);

    virtual void BindKey(unsigned int key, bool bDown, bool bAlt, bool bCtrl, bool bWinKey, bool bAppsKey, bool bShift, const char* pszCommand) = 0;
    virtual void UnbindKey(unsigned int key, bool bDown, bool bAlt, bool bCtrl, bool bWinKey, bool bAppsKey, bool bShift) = 0;
    virtual void UnbindAllKeys(void) = 0;
    virtual bool IsKeyBound(unsigned int key, bool bAlt, bool bCtrl, bool bWinKey, bool bAppsKey, bool bShift) = 0;
    virtual void ListBinds(void) = 0;

    virtual void AddCallback(const char* alias, LPVOID lpGetDataCallback, LPVOID lpGetStateCallback, LPVOID lpKeyboardCallback) = 0;
    virtual void RemoveCallback(const char* alias) = 0;

    virtual unsigned int V2D(unsigned int key) = 0;
    virtual unsigned int D2V(unsigned int key) = 0;
    virtual unsigned int S2D(const char* pszKey) = 0;
    virtual const char* D2S(unsigned int key) = 0;

    virtual bool IsConsoleToggle(unsigned int key, bool bAlt, bool bCtrl, bool bWinKey, bool bAppsKey, bool bShift) = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// IKeyboard
//
// Keyboard interface object to expose helpful functions related to the keyboard.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IMouse : newIDirectInputDevice8A
{
    // The following are exposed from newIDirectInputDevice8A:
    //      HWND GetParentWindow(void);
    //      bool GetBlocked(void);
    //      void SetBlocked(bool bBlocked);
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// IConsole
//
// Console object that allows users to interact with GDHook and its plugins.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IConsole
{
    virtual void Write(const char* message) = 0;
    virtual void Writef(const char* format, ...) = 0;
    virtual void QueueCommand(const char* command) = 0;
    virtual bool HandleCommand(const char* command, bool useCmdQueue) = 0;
    virtual void ProcessCommand(const char* command) = 0;
    virtual void RunTextScript(bool bUseTaskQueue, const char* script) = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// IInputManager
//
// Input manager that maintains the current mouse and keyboard wrappers.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IInputManager
{
    virtual IKeyboard* GetKeyboard(void) = 0;
    virtual IMouse* GetMouse(void) = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// IPluginManager
//
// Plugin manager that maintains the current plugins.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IPluginManager
{
    virtual int GetPluginCount(void) = 0;

    virtual bool LoadPlugin(const char* name) = 0;
    virtual bool UnloadPlugin(const char* name) = 0;
    virtual void* GetPlugin(const char* name) = 0;
    virtual void* GetPlugin(int nIndex) = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// IPointerManager
//
// Pointer manager that maintains the current pointers to various game data.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IPointerManager
{
    virtual unsigned long GetPointer(const char* name) = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// IPrimitiveObject
//
// Primitive object exposure to interact with primitive objects.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IPrimitiveObject
{
    virtual void SetTextureFromFile(const char* path) = 0;
    virtual void SetTextureFromResource(HMODULE hModule, const char* resource) = 0;

    virtual bool GetVisibility(void) const = 0;
    virtual float GetPositionX(void) const = 0;
    virtual float GetPositionY(void) const = 0;
    virtual float GetWidth(void) const = 0;
    virtual float GetHeight(void) const = 0;
    virtual D3DCOLOR GetColor(void) const = 0;
    virtual bool GetBorder(void) const = 0;
    virtual D3DCOLOR GetBorderColor(void) const = 0;

    virtual void SetVisibility(bool bVisible) = 0;
    virtual void SetPositionX(float fPosX) = 0;
    virtual void SetPositionY(float fPosY) = 0;
    virtual void SetWidth(float fWidth) = 0;
    virtual void SetHeight(float fHeight) = 0;
    virtual void SetColor(D3DCOLOR color) = 0;
    virtual void SetBorder(bool bBorder) = 0;
    virtual void SetBorderColor(D3DCOLOR color) = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// IFontObject
//
// Font object exposure to interact with font objects.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IFontObject
{
    virtual const char* GetAlias(void) const = 0;
    virtual D3DCOLOR GetColor(void) const = 0;
    virtual const char* GetFontName(void) const = 0;
    virtual int GetFontHeight(void) const = 0;
    virtual float GetPositionX(void) const = 0;
    virtual float GetPositionY(void) const = 0;
    virtual bool GetLockPosition(void) const = 0;
    virtual const char* GetText(void) const = 0;
    virtual bool GetVisibility(void) const = 0;
    virtual bool GetBold(void) const = 0;
    virtual bool GetItalic(void) const = 0;
    virtual bool GetRightJustified(void) const = 0;
    virtual void GetTextSize(SIZE* lpSize) const = 0;
    virtual GDHook::Enums::FrameAnchor GetAnchor(void) const = 0;
    virtual GDHook::Enums::FrameAnchor GetAnchorParent(void) const = 0;
    virtual bool GetAutoResize(void) const = 0;
    virtual IFontObject* GetParent(void) const = 0;
    virtual MOUSEEVENT GetMouseEventFunction(void) const = 0;

    virtual void SetColor(D3DCOLOR color) = 0;
    virtual void SetFont(const char* font, int nHeight) = 0;
    virtual void SetPosition(float fPosX, float fPosY) = 0;
    virtual void SetLockPosition(bool bLocked) = 0;
    virtual void SetText(const char* text) = 0;
    virtual void SetVisibility(bool bVisible) = 0;
    virtual void SetBold(bool bBold) = 0;
    virtual void SetItalic(bool bItalic) = 0;
    virtual void SetRightJustified(bool bRightJustified) = 0;
    virtual void SetAnchor(GDHook::Enums::FrameAnchor anchor) = 0;
    virtual void SetAnchorParent(GDHook::Enums::FrameAnchor anchor) = 0;
    virtual void SetAutoResize(bool bResize) = 0;
    virtual void SetParent(IFontObject* parent) = 0;
    virtual void SetMouseEventFunction(MOUSEEVENT lpClickFunc) = 0;

    virtual bool HitTest(float fPosX, float fPosY) = 0;

    virtual IPrimitiveObject* GetBackground(void) const = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// IFontManager
//
// Font manager that exposes various font functions.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IFontManager
{
    virtual IFontObject* CreateFontObject(const char* alias) = 0;
    virtual IFontObject* GetFontObject(const char* alias) = 0;
    virtual void DeleteFontObject(const char* alias) = 0;

    virtual bool GetHideObjects(void) const = 0;
    virtual void SetHideObjects(bool bHide) = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// IHookCore
//
// Main GDHook core interface that exposes the various other interfaces.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IHookCore
{
    virtual const char* GetInstallPath(void) const = 0;
    virtual HMODULE GetHandle(void) const = 0;
    virtual HWND GetHwnd(void) const = 0;
    virtual unsigned long GetGameWidth(void) const = 0;
    virtual unsigned long GetGameHeight(void) const = 0;

    virtual IConfigurationManager* GetConfigManager(void) = 0;
    virtual IConsole* GetConsole(void) = 0;
    virtual IFontManager* GetFontManager(void) = 0;
    virtual IInputManager* GetInputManager(void) = 0;
    virtual IPluginManager* GetPluginManager(void) = 0;
    virtual IPointerManager* GetPointerManager(void) = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// PluginData
//
// Plugin data that GDHook plugins expose to the core when being loaded.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
struct PluginData
{
    char    Name[256];          // The name of the plugin.
    char    Author[256];        // The author of the plugin.
    double  InterfaceVersion;   // The interface version this plugin was compiled with.
    double  PluginVersion;      // The version of the plugin.
    int     Priority;           // The priority of the plugin.

    PluginData(void)
    {
        strcpy_s(this->Name, sizeof(this->Name), "Unknown");
        strcpy_s(this->Author, sizeof(this->Author), "atom0s");
        this->InterfaceVersion = GDHOOK_INTERFACE_VERSION;
        this->PluginVersion = 1.00f;
        this->Priority = 0;
    }
    PluginData(const char* name, const char* author, double dblInterfaceVersion, double dblPluginVersion, int nPriority)
    {
        strcpy_s(this->Name, sizeof(this->Name), name);
        strcpy_s(this->Author, sizeof(this->Author), author);
        this->InterfaceVersion = dblInterfaceVersion;
        this->PluginVersion = dblPluginVersion;
        this->Priority = nPriority;
    }
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// IPlugin
//
// Main plugin interface that plugins must follow.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
struct IPlugin
{
    virtual PluginData GetPluginData(void) = 0;

    // Main Plugin Handlers
    virtual bool Initialize(IHookCore* hookCore, DWORD dwPluginId) = 0;
    virtual void Release(void) = 0;

    // Commands
    virtual bool HandleCommand(const char* command) = 0;

    // Direct3D
    virtual bool Direct3DInitialize(IDirect3DDevice9* lpDevice) = 0;
    virtual void Direct3DRelease(void) = 0;
    virtual void Direct3DPreRender(void) = 0;
    virtual void Direct3DRender(void) = 0;
    virtual void Direct3DPreReset(void) = 0;
    virtual void Direct3DPostReset(void) = 0;
    virtual void Direct3DPrePresent(const RECT* pSourceRect, const RECT* pDestRect, HWND hDestWindowOverride, const RGNDATA* pDirtyRegion) = 0;
    virtual void Direct3DPostPresent(const RECT* pSourceRect, const RECT* pDestRect, HWND hDestWindowOverride, const RGNDATA* pDirtyRegion) = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// PluginBase
//
// Main base class that plugins inherit to ensure they follow the proper plugin standard.
//
///////////////////////////////////////////////////////////////////////////////////////////////////
class PluginBase : public IPlugin
{
protected:
    IHookCore*          m_HookCore;         // The main hook core object.
    DWORD               m_PluginId;         // The id of this plugin.
    IDirect3DDevice9*   m_Direct3DDevice9;  // The Direct3D9 device object.

public:
    /**
     * @brief Constructor / Deconstructor
     */
    PluginBase(void) { }
    virtual ~PluginBase(void) { }

    /**
     * @brief Obtains the plugins information.
     *
     * @return The plugin information.
     */
    PluginData GetPluginData(void)
    {
        return PluginData("PluginBase", "atom0s", GDHOOK_INTERFACE_VERSION, 1.0f, 0);
    }

public:
    /**
     * @brief Initializes this plugin and prepares it for use.
     *
     * @param hookCore              The main hook core object exposed to plugins.
     * @param dwPluginId            The id of this plugin. (The base address where it was loaded.)
     *
     * @return True on success, false otherwise. (If false, the plugin will be unloaded.)
     */
    bool Initialize(IHookCore* hookCore, DWORD dwPluginId)
    {
        this->m_HookCore = hookCore;
        this->m_PluginId = dwPluginId;
        return false;
    }

    /**
     * @brief Releases this plugin, called when the plugin is being unloaded.
     */
    void Release(void)
    { }

public:
    /**
     * @brief Handle command call to allow plugins to attempt to handle commands.
     *
     * @param command               The command being processed.
     *
     * @return True if handled, false otherwise.
     */
    bool HandleCommand(const char* command)
    {
        UNREFERENCED_PARAMETER(command);
        return false;
    }

    /**
     * @brief Direct3D initialize call to prepare this plugin for Direct3D calls.
     *
     * @param lpDevice              The Direct3D device currently wrapped.
     *
     * @return True on success, false otherwise.
     *
     * @note    Plugins that do not return true on this call will not receive any other
     *          Direct3D calls listed below!
     */
    bool Direct3DInitialize(IDirect3DDevice9* lpDevice)
    {
        this->m_Direct3DDevice9 = lpDevice;
        return false;
    }

    /**
     * @brief Direct3D release call to allow this plugin to cleanup any Direct3D objects.
     */
    void Direct3DRelease(void)
    { }

    /**
     * @brief Direct3D prerender call to allow this plugin to prepare for rendering.
     */
    void Direct3DPreRender(void)
    { }

    /**
     * @brief Direct3D render call to allow this plugin to render any custom things.
     */
    void Direct3DRender(void)
    { }

    /**
     * @brief Direct3D pre-reset call to allow this plugin to cleanup any objects.
     */
    void Direct3DPreReset(void)
    { }

    /**
     * @brief Direct3D post-reset call to allow this plugin to recreate any objects.
     */
    void Direct3DPostReset(void)
    { }

    /**
     * @brief Direct3D pre-present call to allow this plugin to manipulate a present call.
     */
    void Direct3DPrePresent(const RECT* pSourceRect, const RECT* pDestRect, HWND hDestWindowOverride, const RGNDATA* pDirtyRegion)
    {
        UNREFERENCED_PARAMETER(pSourceRect);
        UNREFERENCED_PARAMETER(pDestRect);
        UNREFERENCED_PARAMETER(hDestWindowOverride);
        UNREFERENCED_PARAMETER(pDirtyRegion);
    }

    /**
     * @brief Direct3D post-present call to allow this plugin to manipulate a present call.
     */
    void Direct3DPostPresent(const RECT* pSourceRect, const RECT* pDestRect, HWND hDestWindowOverride, const RGNDATA* pDirtyRegion)
    {
        UNREFERENCED_PARAMETER(pSourceRect);
        UNREFERENCED_PARAMETER(pDestRect);
        UNREFERENCED_PARAMETER(hDestWindowOverride);
        UNREFERENCED_PARAMETER(pDirtyRegion);
    }
};

///////////////////////////////////////////////////////////////////////////////////////////////////
//
// Plugin Exports
//
// The following are functions that each plugin MUST export in order for it to be able to be 
// loaded within GDHook.
//
// If a plugin does not properly export these functions, it will be rejected from loading.
//
// Plugins must export the following functions:
//      GetInterfaceVersion
//      CreatePluginData
//      CreatePlugin
//
///////////////////////////////////////////////////////////////////////////////////////////////////

typedef double(__stdcall *GetInterfaceVersionFunc)(void);
typedef void(__stdcall *CreatePluginDataFunc)(PluginData* lpBuffer);
typedef IPlugin* (__stdcall *CreatePluginFunc)(char* pszUnused);

#endif // __GDHOOK_SDK_PLUGIN_H_INCLUDED__
